program Images;

uses
  System.Drawing,
  System.Drawing.Text,
  System.Drawing.Imaging,
  System.Drawing.Drawing2D,
  System.Windows.Forms;

type
  { .NET to VCL mapping }
  TApplication = System.Windows.Forms.Application;
  TForm        = System.Windows.Forms.Form;
  TButton      = System.Windows.Forms.Button;

  TSize = System.Drawing.Size;
  TPoint = System.Drawing.Point;

  TGraphics     = System.Drawing.Graphics;
  TBitmap       = System.Drawing.Bitmap;
  TSolidBrush   = System.Drawing.SolidBrush;
  TRectangle    = System.Drawing.Rectangle;
  TFont         = System.Drawing.Font;
  TPen          = System.Drawing.Pen;

type
  { Application main form }
  [STAThread]
  TMainForm = class(TForm)
  private
    Button1: TButton;
    procedure InitComponents;
    procedure Button1Click(Sender: TObject; Args: EventArgs);
  strict protected
    procedure OnPaint(pe: PaintEventArgs); override;
  public
    constructor Create(Owner: TObject);
  end;

{ TMainForm }

constructor TMainForm.Create(Owner: TObject);
begin
  inherited Create;
  InitComponents;
end;

procedure TMainForm.InitComponents;
begin
  SuspendLayout;

  Text:= 'GDI+ Images Samples';
  AutoScaleBaseSize:= TSize.Create(5, 13);
  ClientSize:= TSize.Create(750, 500);

  Button1:= TButton.Create;
  Button1.Top:= 5;
  Button1.Left:= 5;
  Button1.Width:= 100;
  Button1.FlatStyle:= FlatStyle.System;
  Button1.Text:='Draw on Bitmap';
  Button1.Add_Click(Button1Click);
  Controls.Add(Button1);

  ResumeLayout(False);
end;

// Using a a bitmap as a rendering surface is also simple. The following code renders
// some text and lines to a bitmap, and then saves the bitmap to disk as a .png file.
procedure TMainForm.Button1Click(Sender: TObject; Args: EventArgs);
var
  g: TGraphics;
  newBitmap: TBitmap;
  textFont: TFont;
  rectangle: RectangleF;
  flowedText: string;
  penExample: TPen;
  points: array[0..4] of TPoint;
begin
  newBitmap:= TBitmap.Create(800,600,PixelFormat.Format32bppArgb);
  g:= Graphics.FromImage(newBitmap);
  g.FillRectangle(TSolidBrush.Create(Color.White), TRectangle.Create(0,0,800,600));

  textFont:= TFont.Create('Lucida Sans Unicode', 12);
  rectangle:= RectangleF.Create(100, 100, 250, 350);

  flowedText:= 'Simplicity and power: Windows Forms is a programming model for'#10
    + 'developing Windows applications that combines the simplicity of the Delphi''s'#10
    + 'VCL.NET programming model with the power and flexibility of the common'#10
    + 'language runtime.'#10
    + 'Lower total cost of ownership: Windows Forms takes advantage of the versioning and'#10
    + 'deployment features of the common language runtime to offer reduced deployment'#10
    + 'costs and higher application robustness over time. This significantly lowers the'#10
    + 'maintenance costs (TCO) for applications'#10
    + 'written in Windows Forms.'#10
    + 'Architecture for controls: Windows Forms offers an architecture for'#10
    + 'controls and control containers that is based on concrete implementation of the'#10
    + 'control and container classes. This significantly reduces'#10
    + 'control-container interoperability issues.';

  g.FillRectangle(TSolidBrush.Create(Color.Gainsboro), rectangle);
  g.DrawString(flowedText, textFont, TSolidBrush.Create(Color.Blue), rectangle);
  penExample:= TPen.Create(Color.FromArgb(150, Color.Purple), 20);
  penExample.DashStyle:= DashStyle.Dash;
  penExample.StartCap:= LineCap.Round;
  penExample.EndCap:= LineCap.Round;


  // Create the path of the curve with an array of points.
  points[0]:= TPoint.Create(200, 140);
  points[1]:= TPoint.Create(600, 240);
  points[2]:= TPoint.Create(500, 340);
  points[3]:= TPoint.Create(140, 140);
  points[4]:= TPoint.Create(40, 340);

  g.DrawCurve(penExample, points);

  newBitmap.Save('TestImage.png', ImageFormat.Png) ;
  MessageBox.Show('Image file created!');
end;

// GDI+ has full support for a range of image formats, including:
// .jpeg, .png, .gif, .bmp, .tiff, .exif, and .icon files.
// Rendering an image is simple. For instance, the following renders a .jpeg image.
procedure TMainForm.OnPaint(pe: PaintEventArgs);
var
  b: TBitmap;
  c: Color;
begin
  b:= TBitmap.Create('splash1.jpg');

  pe.Graphics.FillRectangle(TSolidBrush.Create(Color.White), ClientRectangle);

  b.MakeTransparent(BackColor);
  pe.Graphics.DrawImage(b, 50, 50, 450, 350);

  pe.Graphics.FillRectangle(TSolidBrush.Create(Color.White), ClientRectangle);
  b:= TBitmap.Create('splash1.jpg');
  c:= b.GetPixel(1,1);

  b.MakeTransparent(c);
  pe.Graphics.DrawImage(b, 50, 50, 450, 350);
end;


{
  Note:
  The graphics object that is contained in the arguments to the Paint event
  (PaintEventArgs) is disposed upon return from the Paint event handler.
  Therefore, you should not keep a reference to this Graphics object that
  has scope outside the Paint event. Attempting to use this Graphics object
  outside the Paint event will have unpredictable results.

  By default, the Paint event is not raised when a control or form is resized.
  If you want the Paint event to be raised when a form is resized, you need to
  set the control style appropriately.

  ex.: SetStyle(ControlStyles.ResizeRedraw,true);

}

var
  MainForm: TMainForm;
begin
  MainForm:= TMainForm.Create(nil);
  Application.Run(MainForm);
  MainForm.Free;
end.
